/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.util;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.invoke.CallSite;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.apache.commons.io.FileUtils;
import org.geoserver.platform.resource.Files;
import org.geotools.util.logging.Logging;

public class IOUtils {
    private static final int DEFAULT_BUFFER_SIZE = 16384;
    private static final Logger LOGGER = Logging.getLogger(IOUtils.class);

    protected IOUtils() {
    }

    public static void copy(InputStream from, File to) throws IOException {
        IOUtils.copy(from, new FileOutputStream(to), 16384);
    }

    public static void copy(InputStream in, OutputStream out) throws IOException {
        IOUtils.copy(in, out, 16384);
    }

    public static void copy(InputStream in, OutputStream out, int copyBufferSize) throws IOException {
        try (InputStream inputStream = in;
             OutputStream outputStream = out;){
            byte[] buffer = new byte[copyBufferSize];
            int bytes = 0;
            while ((bytes = in.read(buffer)) != -1) {
                out.write(buffer, 0, bytes);
            }
            out.flush();
        }
    }

    public static void filteredCopy(File from, File to, Map<String, String> filters) throws IOException {
        IOUtils.filteredCopy(new BufferedReader(new FileReader(from)), to, filters);
    }

    public static String toString(InputStream input) throws IOException {
        String toReturn;
        if (input == null) {
            return null;
        }
        try (InputStream inputStream = input;
             ByteArrayOutputStream output = new ByteArrayOutputStream();){
            IOUtils.copy(input, output);
            toReturn = output.toString();
        }
        return toReturn;
    }

    public static void filteredCopy(BufferedReader from, File to, Map<String, String> filters) throws IOException {
        HashMap<CallSite, String> escapedMap = new HashMap<CallSite, String>();
        for (Map.Entry<String, String> entry : filters.entrySet()) {
            escapedMap.put((CallSite)((Object)("${" + entry.getKey() + "}")), entry.getValue());
        }
        try (BufferedReader bufferedReader = from;
             BufferedWriter out = new BufferedWriter(new FileWriter(to));){
            String line = null;
            while ((line = from.readLine()) != null) {
                for (Map.Entry entry : escapedMap.entrySet()) {
                    line = line.replace((CharSequence)entry.getKey(), (CharSequence)entry.getValue());
                }
                out.write(line);
                out.newLine();
            }
            out.flush();
        }
    }

    public static void copy(File from, File to) throws IOException {
        IOUtils.copy((InputStream)new FileInputStream(from), to);
    }

    public static void deepCopy(File fromDir, File toDir) throws IOException {
        if (!fromDir.isDirectory() || !fromDir.exists()) {
            throw new IllegalArgumentException("Invalid source directory (it's either not a directory, or does not exist");
        }
        if (toDir.exists() && toDir.isFile()) {
            throw new IllegalArgumentException("Invalid destination directory, it happens to be a file instead");
        }
        if (!toDir.exists() && !toDir.mkdir()) {
            throw new IOException("Could not create " + String.valueOf(toDir));
        }
        File[] files = fromDir.listFiles();
        if (files != null) {
            for (File file : files) {
                File destination = new File(toDir, file.getName());
                if (file.isDirectory()) {
                    IOUtils.deepCopy(file, destination);
                    continue;
                }
                IOUtils.copy(file, destination);
            }
        }
    }

    public static File createRandomDirectory(String baseDir, String prefix, String suffix) throws IOException {
        File tempDir = File.createTempFile(prefix, suffix, new File(baseDir));
        tempDir.delete();
        if (!tempDir.mkdir()) {
            throw new IOException("Could not create the temp directory " + tempDir.getPath());
        }
        return tempDir;
    }

    public static File createTempDirectory(String prefix) throws IOException {
        File dummyTemp = File.createTempFile("blah", null);
        String sysTempDir = dummyTemp.getParentFile().getAbsolutePath();
        dummyTemp.delete();
        File reqTempDir = new File(sysTempDir + File.separator + prefix + ThreadLocalRandom.current().nextDouble());
        reqTempDir.mkdir();
        return reqTempDir;
    }

    public static boolean delete(File directory) throws IOException {
        return IOUtils.emptyDirectory(directory, false);
    }

    public static boolean delete(File directory, boolean quiet) throws IOException {
        try {
            IOUtils.emptyDirectory(directory, quiet);
            return directory.delete();
        }
        catch (Exception exception) {
            if (!quiet) {
                throw exception;
            }
            return false;
        }
    }

    public static boolean emptyDirectory(File directory) throws IOException {
        return IOUtils.emptyDirectory(directory, false);
    }

    public static boolean emptyDirectory(File directory, boolean quiet) throws IOException {
        if (!directory.isDirectory()) {
            if (!quiet) {
                throw new IllegalArgumentException("The provide file '%s' doesn't appear to be a directory.".formatted(directory.getAbsolutePath()));
            }
            return false;
        }
        File[] files = directory.listFiles();
        if (files == null) {
            if (!quiet) {
                throw new IllegalStateException("Not able to list files of '%s'.".formatted(directory.getAbsolutePath()));
            }
            return false;
        }
        boolean allClean = true;
        for (File file : files) {
            if (file.isDirectory()) {
                allClean &= IOUtils.delete(file);
                continue;
            }
            if (file.delete()) continue;
            if (!quiet) {
                LOGGER.log(Level.WARNING, "Could not delete {0}", file.getAbsolutePath());
            }
            allClean = false;
        }
        return allClean;
    }

    public static void zipDirectory(File directory, ZipOutputStream zipout, FilenameFilter filter) throws IOException, FileNotFoundException {
        IOUtils.zipDirectory(directory, "", zipout, filter);
    }

    private static void zipDirectory(File directory, String prefix, ZipOutputStream zipout, FilenameFilter filter) throws IOException, FileNotFoundException {
        File[] files = directory.listFiles(filter);
        byte[] buffer = new byte[4096];
        if (files != null) {
            for (File file : files) {
                if (!file.exists()) continue;
                if (file.isDirectory()) {
                    String newPrefix = prefix + file.getName() + "/";
                    zipout.putNextEntry(new ZipEntry(newPrefix));
                    IOUtils.zipDirectory(file, newPrefix, zipout, filter);
                    continue;
                }
                ZipEntry entry = new ZipEntry(prefix + file.getName());
                zipout.putNextEntry(entry);
                try (FileInputStream in = new FileInputStream(file);){
                    int c;
                    while (-1 != (c = ((InputStream)in).read(buffer))) {
                        zipout.write(buffer, 0, c);
                    }
                    zipout.closeEntry();
                }
            }
        }
        zipout.flush();
    }

    public static File getZipOutputFile(File destDir, ZipEntry entry) throws IOException {
        String canonicalDirectory = destDir.getCanonicalPath();
        File file = new File(destDir, entry.getName());
        String canonicalFile = file.getCanonicalPath();
        if (canonicalFile.startsWith(canonicalDirectory + File.separator)) {
            return file;
        }
        throw new IOException("Entry is outside of the target directory: " + entry.getName());
    }

    public static void decompress(InputStream input, File destDir) throws IOException {
        ZipInputStream zin = new ZipInputStream(input);
        ZipEntry entry = null;
        byte[] buffer = new byte[1024];
        while ((entry = zin.getNextEntry()) != null) {
            File f = IOUtils.getZipOutputFile(destDir, entry);
            if (entry.isDirectory()) {
                f.mkdirs();
                continue;
            }
            try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(f));){
                int n = -1;
                while ((n = zin.read(buffer)) != -1) {
                    out.write(buffer, 0, n);
                }
                out.flush();
            }
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static void decompress(File inputFile, File destDir) throws IOException {
        try (ZipFile zipFile = new ZipFile(inputFile);){
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                File newFile = IOUtils.getZipOutputFile(destDir, entry);
                if (entry.isDirectory()) {
                    newFile.mkdir();
                    continue;
                }
                try {
                    InputStream stream = zipFile.getInputStream(entry);
                    try (FileOutputStream fos = new FileOutputStream(newFile);){
                        int len;
                        byte[] buf = new byte[1024];
                        while ((len = stream.read(buf)) >= 0) {
                            IOUtils.saveCompressedStream(buf, fos, len);
                        }
                        fos.flush();
                    }
                    finally {
                        if (stream == null) continue;
                        stream.close();
                    }
                }
                catch (IOException e) {
                    IOException ioe = new IOException("Not valid archive file type.");
                    ioe.initCause(e);
                    throw ioe;
                    return;
                }
            }
        }
    }

    public static void saveCompressedStream(byte[] buffer, OutputStream out, int len) throws IOException {
        try {
            out.write(buffer, 0, len);
        }
        catch (Exception e) {
            out.flush();
            out.close();
            IOException ioe = new IOException("Not valid archive file type.");
            ioe.initCause(e);
            throw ioe;
        }
    }

    public static void backupDirectory(File dir) throws IOException {
        File bak = new File(dir.getCanonicalPath() + ".bak");
        if (bak.exists()) {
            FileUtils.deleteDirectory((File)bak);
        }
        dir.renameTo(bak);
    }

    public static void rename(File f, String newName) throws IOException {
        IOUtils.rename(f, new File(f.getParentFile(), newName));
    }

    public static void rename(File source, File dest) throws IOException {
        if (source.getCanonicalPath().equals(dest.getCanonicalPath())) {
            return;
        }
        boolean win = System.getProperty("os.name").startsWith("Windows");
        if (win && dest.exists() && !dest.delete()) {
            throw new IOException("Could not delete: " + dest.getCanonicalPath());
        }
        if (!Files.move(source, dest)) {
            FileUtils.deleteQuietly((File)dest);
            if (source.isDirectory()) {
                FileUtils.moveDirectory((File)source, (File)dest);
            } else {
                FileUtils.moveFile((File)source, (File)dest);
            }
        }
    }

    public static void closeQuietly(Closeable closeable) {
        try {
            if (closeable != null) {
                closeable.close();
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }
}

