/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.referencing;

import java.awt.Shape;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.text.Format;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import net.sf.geographiclib.Geodesic;
import net.sf.geographiclib.GeodesicData;
import net.sf.geographiclib.GeodesicLine;
import org.geotools.api.geometry.Position;
import org.geotools.api.referencing.crs.CompoundCRS;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.api.referencing.crs.GeographicCRS;
import org.geotools.api.referencing.cs.AxisDirection;
import org.geotools.api.referencing.cs.CoordinateSystem;
import org.geotools.api.referencing.cs.CoordinateSystemAxis;
import org.geotools.api.referencing.cs.EllipsoidalCS;
import org.geotools.api.referencing.datum.Datum;
import org.geotools.api.referencing.datum.Ellipsoid;
import org.geotools.api.referencing.datum.GeodeticDatum;
import org.geotools.api.referencing.datum.PrimeMeridian;
import org.geotools.api.referencing.operation.TransformException;
import org.geotools.api.util.Cloneable;
import org.geotools.geometry.GeneralPosition;
import org.geotools.geometry.Position2D;
import org.geotools.geometry.TransformedPosition;
import org.geotools.measure.Angle;
import org.geotools.measure.Latitude;
import org.geotools.measure.Longitude;
import org.geotools.metadata.i18n.Vocabulary;
import org.geotools.referencing.CRS;
import org.geotools.referencing.CoordinateFormat;
import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.geotools.referencing.cs.DefaultEllipsoidalCS;
import org.geotools.referencing.datum.DefaultEllipsoid;
import org.geotools.referencing.datum.DefaultGeodeticDatum;
import org.geotools.referencing.datum.DefaultPrimeMeridian;
import org.geotools.referencing.util.CRSUtilities;
import org.geotools.util.TableWriter;
import si.uom.NonSI;

public class GeodeticCalculator {
    private final TransformedPosition userToGeodetic;
    private CoordinateReferenceSystem coordinateReferenceSystem;
    private GeographicCRS geographicCRS;
    private final Ellipsoid ellipsoid;
    private final double semiMajorAxis;
    private final double flattening;
    private double lat1;
    private double long1;
    private double lat2;
    private double long2;
    private double distance;
    private double azimuth;
    private boolean destinationValid;
    private boolean directionValid;
    private Geodesic geod;

    public GeodeticCalculator() {
        this(DefaultEllipsoid.WGS84);
    }

    public GeodeticCalculator(Ellipsoid ellipsoid) {
        this(ellipsoid, null);
    }

    public GeodeticCalculator(CoordinateReferenceSystem crs) {
        this(CRS.getEllipsoid(crs), crs);
    }

    private GeodeticCalculator(Ellipsoid ellipsoid, CoordinateReferenceSystem crs) {
        if (ellipsoid == null) {
            throw new IllegalArgumentException(MessageFormat.format("Argument \"{0}\" should not be null.", "ellipsoid"));
        }
        this.ellipsoid = ellipsoid;
        this.semiMajorAxis = ellipsoid.getSemiMajorAxis();
        this.flattening = 1.0 / ellipsoid.getInverseFlattening();
        this.geod = new Geodesic(this.semiMajorAxis, this.flattening);
        if (crs != null) {
            this.coordinateReferenceSystem = crs;
            this.geographicCRS = GeodeticCalculator.getGeographicCRS(crs);
            this.userToGeodetic = new TransformedPosition(crs, (CoordinateReferenceSystem)this.geographicCRS, null);
        } else {
            this.userToGeodetic = null;
        }
    }

    private static GeographicCRS getGeographicCRS(CoordinateReferenceSystem crs) {
        Datum datum;
        if (crs instanceof GeographicCRS) {
            GeographicCRS rS = (GeographicCRS)crs;
            CoordinateSystem cs = crs.getCoordinateSystem();
            if (cs.getDimension() == 2 && GeodeticCalculator.isStandard(cs.getAxis(0), AxisDirection.EAST) && GeodeticCalculator.isStandard(cs.getAxis(1), AxisDirection.NORTH)) {
                return rS;
            }
        }
        if ((datum = CRSUtilities.getDatum(crs)) instanceof GeodeticDatum) {
            GeodeticDatum geodeticDatum = (GeodeticDatum)datum;
            return new DefaultGeographicCRS("Geodetic", geodeticDatum, (EllipsoidalCS)DefaultEllipsoidalCS.GEODETIC_2D);
        }
        if (crs instanceof CompoundCRS) {
            CompoundCRS rS = (CompoundCRS)crs;
            for (CoordinateReferenceSystem component : rS.getCoordinateReferenceSystems()) {
                GeographicCRS candidate = GeodeticCalculator.getGeographicCRS(component);
                if (candidate == null) continue;
                return candidate;
            }
        }
        throw new IllegalArgumentException("Illegal coordinate reference system.");
    }

    private static boolean isStandard(CoordinateSystemAxis axis, AxisDirection direction) {
        return direction.equals((Object)axis.getDirection()) && NonSI.DEGREE_ANGLE.equals(axis.getUnit());
    }

    private static void checkLatitude(double latitude) throws IllegalArgumentException {
        if (!(latitude >= -90.0) || !(latitude <= 90.0)) {
            Latitude arg0 = new Latitude(latitude);
            throw new IllegalArgumentException(MessageFormat.format("Latitude {0} is out of range (\u00b190\u00b0).", arg0));
        }
    }

    private static void checkLongitude(double longitude) throws IllegalArgumentException {
        if (!(Math.abs(longitude) <= Double.MAX_VALUE)) {
            Longitude arg1 = new Longitude(longitude);
            throw new IllegalArgumentException(MessageFormat.format("Illegal argument: \"{0}={1}\".", "longitude", arg1));
        }
    }

    private static void checkAzimuth(double azimuth) throws IllegalArgumentException {
        if (!(Math.abs(azimuth) <= Double.MAX_VALUE)) {
            Longitude arg1 = new Longitude(azimuth);
            throw new IllegalArgumentException(MessageFormat.format("Illegal argument: \"{0}={1}\".", "azimuth", arg1));
        }
    }

    private static void checkOrthodromicDistance(double distance) throws IllegalArgumentException {
        if (!(Math.abs(distance) <= Double.MAX_VALUE)) {
            throw new IllegalArgumentException(MessageFormat.format("Illegal argument: \"{0}={1}\".", "distance", distance));
        }
    }

    private static String format(Format cf, double longitude, double latitude) {
        return cf.format(new GeneralPosition(longitude, latitude));
    }

    public CoordinateReferenceSystem getCoordinateReferenceSystem() {
        if (this.coordinateReferenceSystem == null) {
            this.coordinateReferenceSystem = this.getGeographicCRS();
        }
        return this.coordinateReferenceSystem;
    }

    public GeographicCRS getGeographicCRS() {
        if (this.geographicCRS == null) {
            String name = Vocabulary.format((int)83);
            this.geographicCRS = new DefaultGeographicCRS(name, (GeodeticDatum)new DefaultGeodeticDatum(name, this.getEllipsoid(), (PrimeMeridian)DefaultPrimeMeridian.GREENWICH), (EllipsoidalCS)DefaultEllipsoidalCS.GEODETIC_2D);
        }
        return this.geographicCRS;
    }

    public Ellipsoid getEllipsoid() {
        return this.ellipsoid;
    }

    public void setStartingGeographicPoint(double longitude, double latitude) throws IllegalArgumentException {
        GeodeticCalculator.checkLongitude(longitude);
        GeodeticCalculator.checkLatitude(latitude);
        this.long1 = longitude;
        this.lat1 = latitude;
        this.destinationValid = false;
        this.directionValid = false;
    }

    public void setStartingGeographicPoint(Point2D point) throws IllegalArgumentException {
        this.setStartingGeographicPoint(point.getX(), point.getY());
    }

    public void setStartingPosition(Position position) throws TransformException {
        if (this.userToGeodetic != null) {
            this.userToGeodetic.transform(position);
            position = this.userToGeodetic;
        }
        this.setStartingGeographicPoint(position.getOrdinate(0), position.getOrdinate(1));
    }

    public Point2D getStartingGeographicPoint() {
        return new Point2D.Double(this.long1, this.lat1);
    }

    public Position getStartingPosition() throws TransformException {
        Cloneable position = this.userToGeodetic;
        if (position == null) {
            position = new Position2D();
        }
        position.setOrdinate(0, this.long1);
        position.setOrdinate(1, this.lat1);
        if (this.userToGeodetic != null) {
            position = this.userToGeodetic.inverseTransform();
        }
        return position;
    }

    public void setDestinationGeographicPoint(double longitude, double latitude) throws IllegalArgumentException {
        GeodeticCalculator.checkLongitude(longitude);
        GeodeticCalculator.checkLatitude(latitude);
        this.long2 = longitude;
        this.lat2 = latitude;
        this.destinationValid = true;
        this.directionValid = false;
    }

    public void setDestinationGeographicPoint(Point2D point) throws IllegalArgumentException {
        this.setDestinationGeographicPoint(point.getX(), point.getY());
    }

    public void setDestinationPosition(Position position) throws TransformException {
        if (this.userToGeodetic != null) {
            this.userToGeodetic.transform(position);
            position = this.userToGeodetic;
        }
        this.setDestinationGeographicPoint(position.getOrdinate(0), position.getOrdinate(1));
    }

    public Point2D getDestinationGeographicPoint() throws IllegalStateException {
        if (!this.destinationValid) {
            this.computeDestinationPoint();
        }
        return new Point2D.Double(this.long2, this.lat2);
    }

    public Position getDestinationPosition() throws TransformException {
        Cloneable position;
        if (!this.destinationValid) {
            this.computeDestinationPoint();
        }
        if ((position = this.userToGeodetic) == null) {
            position = new Position2D();
        }
        position.setOrdinate(0, this.long2);
        position.setOrdinate(1, this.lat2);
        if (this.userToGeodetic != null) {
            position = this.userToGeodetic.inverseTransform();
        }
        return position;
    }

    public void setDirection(double azimuth, double distance) throws IllegalArgumentException {
        GeodeticCalculator.checkAzimuth(azimuth);
        GeodeticCalculator.checkOrthodromicDistance(distance);
        this.azimuth = azimuth;
        this.distance = distance;
        this.destinationValid = false;
        this.directionValid = true;
    }

    public double getAzimuth() throws IllegalStateException {
        if (!this.directionValid) {
            this.computeDirection();
        }
        return this.azimuth;
    }

    public double getOrthodromicDistance() throws IllegalStateException {
        if (!this.directionValid) {
            this.computeDirection();
        }
        return this.distance;
    }

    private void computeDestinationPoint() throws IllegalStateException {
        if (!this.directionValid) {
            throw new IllegalStateException("The direction has not been set.");
        }
        GeodesicData g = this.geod.Direct(this.lat1, this.long1, this.azimuth, this.distance);
        this.lat2 = g.lat2;
        this.long2 = g.lon2;
        this.destinationValid = true;
    }

    public double getMeridianArcLength(double latitude1, double latitude2) {
        GeodeticCalculator.checkLatitude(latitude1);
        GeodeticCalculator.checkLatitude(latitude2);
        GeodesicData g = this.geod.Inverse(latitude1, 0.0, latitude2, 0.0, 1025);
        return g.s12;
    }

    private void computeDirection() throws IllegalStateException {
        if (!this.destinationValid) {
            throw new IllegalStateException("The destination has not been set.");
        }
        GeodesicData g = this.geod.Inverse(this.lat1, this.long1, this.lat2, this.long2);
        this.azimuth = g.azi1;
        this.distance = g.s12;
        this.directionValid = true;
    }

    public Shape getGeodeticCurve(int numberOfPoints) {
        List<Point2D> points = this.getGeodeticPath(numberOfPoints);
        GeneralPath path = new GeneralPath(0, numberOfPoints + 1);
        Point2D start = points.get(0);
        path.moveTo(start.getX(), start.getY());
        for (int i = 1; i < points.size(); ++i) {
            Point2D p = points.get(i);
            path.lineTo(p.getX(), p.getY());
        }
        return path;
    }

    public Shape getGeodeticCurve() {
        return this.getGeodeticCurve(10);
    }

    public List<Point2D> getGeodeticPath(int numPoints) {
        if (numPoints < 0) {
            throw new IllegalArgumentException(MessageFormat.format("Illegal argument: \"{0}={1}\".", "numPoints", numPoints));
        }
        ArrayList<Point2D> points = new ArrayList<Point2D>(numPoints + 2);
        if (!this.directionValid) {
            this.computeDirection();
        }
        if (!this.destinationValid) {
            this.computeDestinationPoint();
        }
        double delta = this.distance / (double)(numPoints + 1);
        points.add(new Point2D.Double(this.long1, this.lat1));
        GeodesicLine line = this.geod.Line(this.lat1, this.long1, this.azimuth);
        for (int i = 1; i <= numPoints + 1; ++i) {
            GeodesicData g = line.Position((double)i * delta, 33160);
            points.add(new Point2D.Double(g.lon2, g.lat2));
        }
        return points;
    }

    public String toString() {
        Vocabulary resources = Vocabulary.getResources(null);
        TableWriter buffer = new TableWriter(null, " ");
        if (this.coordinateReferenceSystem != null) {
            buffer.write(resources.getLabel(32));
            buffer.nextColumn();
            buffer.write(this.coordinateReferenceSystem.getName().getCode());
            buffer.nextLine();
        }
        if (this.ellipsoid != null) {
            buffer.write(resources.getLabel(56));
            buffer.nextColumn();
            buffer.write(this.ellipsoid.getName().getCode());
            buffer.nextLine();
        }
        CoordinateFormat cf = new CoordinateFormat();
        Format nf = cf.getFormat(0);
        buffer.write(resources.getLabel(201));
        buffer.nextColumn();
        buffer.write(GeodeticCalculator.format(cf, this.long1, this.lat1));
        buffer.nextLine();
        if (this.destinationValid) {
            buffer.write(resources.getLabel(212));
            buffer.nextColumn();
            buffer.write(GeodeticCalculator.format(cf, this.long2, this.lat2));
            buffer.nextLine();
        }
        if (this.directionValid) {
            buffer.write(resources.getLabel(8));
            buffer.nextColumn();
            buffer.write(nf.format(new Angle(this.azimuth)));
            buffer.nextLine();
        }
        if (this.directionValid) {
            buffer.write(resources.getLabel(159));
            buffer.nextColumn();
            buffer.write(nf.format(this.distance));
            if (this.ellipsoid != null) {
                buffer.write(32);
                buffer.write(this.ellipsoid.getAxisUnit().toString());
            }
            buffer.nextLine();
        }
        return buffer.toString();
    }
}

