/*
 * Decompiled with CFR 0.152.
 */
package org.geoserver.wms.svg;

import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;
import java.util.logging.Logger;
import org.geotools.api.data.DataSourceException;
import org.geotools.api.feature.simple.SimpleFeature;
import org.geotools.api.feature.simple.SimpleFeatureType;
import org.geotools.data.simple.SimpleFeatureIterator;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

class SVGWriter
extends OutputStreamWriter {
    private static final Logger LOGGER = Logging.getLogger((String)SVGWriter.class.getPackage().getName());
    private static DecimalFormat formatter;
    private HashMap<Class<? extends Geometry>, SVGFeatureWriter> writers;
    private double minY;
    private double maxY;
    private SVGFeatureWriterHandler writerHandler = new SVGFeatureWriterHandler();
    private SVGFeatureWriter featureWriter = null;
    private double minCoordDistance;
    private boolean pointsAsCircles;

    public SVGWriter(OutputStream out, Envelope mapAreaOfInterest) {
        super(out);
        this.minY = mapAreaOfInterest.getMinY();
        this.maxY = mapAreaOfInterest.getMaxY();
        this.initWriters();
    }

    private void initWriters() {
        this.writers = new HashMap();
        this.writers.put(Point.class, new PointWriter());
        this.writers.put(LineString.class, new LineStringWriter());
        this.writers.put(LinearRing.class, new LineStringWriter());
        this.writers.put(Polygon.class, new PolygonWriter());
        this.writers.put(MultiPoint.class, new MultiPointWriter());
        this.writers.put(MultiLineString.class, new MultiLineStringWriter());
        this.writers.put(MultiPolygon.class, new MultiPolygonWriter());
    }

    public void setPointsAsCircles(boolean asCircles) {
        this.pointsAsCircles = asCircles;
    }

    public void setGeometryType(Class<?> gtype) {
        this.featureWriter = this.writers.get(gtype);
        if (this.featureWriter == null) {
            if (gtype == Geometry.class) {
                this.featureWriter = new GeometryWriter();
            } else {
                throw new IllegalArgumentException("No SVG Feature writer defined for " + String.valueOf(gtype));
            }
        }
    }

    public void setWriterHandler(SVGFeatureWriterHandler handler) {
        this.writerHandler = handler;
    }

    public void setMinCoordDistance(double minCoordDistance) {
        this.minCoordDistance = minCoordDistance;
    }

    public double getY(double y) {
        return this.maxY - y + this.minY;
    }

    public double getX(double x) {
        return x;
    }

    public void setMaximunFractionDigits(int numDigits) {
        formatter.setMaximumFractionDigits(numDigits);
    }

    public int getMaximunFractionDigits() {
        return formatter.getMaximumFractionDigits();
    }

    public void setMinimunFractionDigits(int numDigits) {
        formatter.setMinimumFractionDigits(numDigits);
    }

    public int getMinimunFractionDigits() {
        return formatter.getMinimumFractionDigits();
    }

    public void write(double d) throws IOException {
        this.write(formatter.format(d));
    }

    public void write(char c) throws IOException {
        super.write(c);
    }

    public void newline() throws IOException {
        super.write(10);
    }

    public void writeFeatures(SimpleFeatureType featureType, SimpleFeatureIterator reader, String style) throws IOException {
        try {
            Class gtype = featureType.getGeometryDescriptor().getType().getBinding();
            boolean doCollect = false;
            this.setGeometryType(gtype);
            this.setPointsAsCircles("#circle".equals(style));
            this.setUpWriterHandler(featureType, doCollect);
            if (doCollect) {
                this.write("<path ");
                this.write("d=\"");
            }
            while (reader.hasNext()) {
                SimpleFeature ft = (SimpleFeature)reader.next();
                this.writeFeature(ft);
                ft = null;
            }
            if (doCollect) {
                this.write("\"/>\n");
            }
            LOGGER.fine("encoded " + featureType.getTypeName());
        }
        catch (Exception ex) {
            throw new DataSourceException(ex.getMessage(), (Throwable)ex);
        }
    }

    private void setUpWriterHandler(SimpleFeatureType featureType, boolean doCollect) throws IOException {
        if (doCollect) {
            this.writerHandler = new CollectSVGHandler(this.featureWriter);
            LOGGER.finer("Established a collecting features writer handler");
        } else {
            this.writerHandler = new SVGFeatureWriterHandler();
            ArrayList atts = new ArrayList(0);
            if (atts.contains("#FID")) {
                this.writerHandler = new FIDSVGHandler(this.writerHandler);
                atts.remove("#FID");
                LOGGER.finer("Added FID handler decorator");
            }
            if (atts.contains("#BOUNDS")) {
                this.writerHandler = new BoundsSVGHandler(this.writerHandler);
                atts.remove("#BOUNDS");
                LOGGER.finer("Added BOUNDS handler decorator");
            }
            if (!atts.isEmpty()) {
                this.writerHandler = new AttributesSVGHandler(this.writerHandler);
                LOGGER.finer("Added ATTRIBUTES handler decorator");
            }
        }
    }

    public void writeFeature(SimpleFeature ft) throws IOException {
        this.writerHandler.startFeature(this.featureWriter, ft);
        this.writerHandler.startGeometry(this.featureWriter, ft);
        this.writerHandler.writeGeometry(this.featureWriter, ft);
        this.writerHandler.endGeometry(this.featureWriter, ft);
        this.writerHandler.endFeature(this.featureWriter, ft);
    }

    static {
        Locale locale = new Locale("en", "US");
        DecimalFormatSymbols decimalSymbols = new DecimalFormatSymbols(locale);
        decimalSymbols.setDecimalSeparator('.');
        formatter = new DecimalFormat();
        formatter.setDecimalFormatSymbols(decimalSymbols);
        formatter.setGroupingSize(0);
        formatter.setDecimalSeparatorAlwaysShown(false);
        formatter.setDecimalFormatSymbols(DecimalFormatSymbols.getInstance(Locale.ENGLISH));
        formatter.setMaximumFractionDigits(5);
        formatter.setMinimumFractionDigits(0);
    }

    public static class SVGFeatureWriterHandler {
        public void startFeature(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            featureWriter.startElement(ft);
        }

        public void endFeature(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            featureWriter.endElement(ft);
        }

        public void startGeometry(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            featureWriter.startGeometry((Geometry)ft.getDefaultGeometry());
        }

        public void writeGeometry(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            featureWriter.writeGeometry((Geometry)ft.getDefaultGeometry());
        }

        public void endGeometry(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            featureWriter.endGeometry((Geometry)ft.getDefaultGeometry());
        }
    }

    private abstract class SVGFeatureWriter {
        private SVGFeatureWriter() {
        }

        protected abstract void startElement(SimpleFeature var1) throws IOException;

        protected abstract void startGeometry(Geometry var1) throws IOException;

        protected abstract void writeGeometry(Geometry var1) throws IOException;

        protected void endGeometry(Geometry geom) throws IOException {
            SVGWriter.this.write("\"");
        }

        protected void endElement(SimpleFeature feature) throws IOException {
            SVGWriter.this.write("/>\n");
        }

        protected void writePathContent(Coordinate[] coords) throws IOException {
            SVGWriter.this.write('M');
            Coordinate prev = coords[0];
            Coordinate curr = null;
            SVGWriter.this.write(SVGWriter.this.getX(prev.x));
            SVGWriter.this.write(' ');
            SVGWriter.this.write(SVGWriter.this.getY(prev.y));
            int nCoords = coords.length;
            SVGWriter.this.write('l');
            for (int i = 1; i < nCoords; ++i) {
                curr = coords[i];
                if (i > 3 && prev.distance(curr) <= SVGWriter.this.minCoordDistance) continue;
                SVGWriter.this.write(SVGWriter.this.getX(curr.x) - SVGWriter.this.getX(prev.x));
                SVGWriter.this.write(' ');
                SVGWriter.this.write(SVGWriter.this.getY(curr.y) - SVGWriter.this.getY(prev.y));
                SVGWriter.this.write(' ');
                prev = curr;
            }
        }

        protected void writeClosedPathContent(Coordinate[] coords) throws IOException {
            this.writePathContent(coords);
            SVGWriter.this.write('Z');
        }
    }

    private class PointWriter
    extends SVGFeatureWriter {
        @Override
        protected void startElement(SimpleFeature feature) throws IOException {
            SVGWriter.this.write(SVGWriter.this.pointsAsCircles ? "<circle r='0.25%' fill='blue'" : "<use");
        }

        @Override
        protected void startGeometry(Geometry geom) throws IOException {
        }

        protected void writeBounds(Envelope env) throws IOException {
        }

        @Override
        protected void writeGeometry(Geometry geom) throws IOException {
            Point p = (Point)geom;
            if (SVGWriter.this.pointsAsCircles) {
                SVGWriter.this.write(" cx=\"");
                SVGWriter.this.write(SVGWriter.this.getX(p.getX()));
                SVGWriter.this.write("\" cy=\"");
                SVGWriter.this.write(SVGWriter.this.getY(p.getY()));
            } else {
                SVGWriter.this.write(" x=\"");
                SVGWriter.this.write(SVGWriter.this.getX(p.getX()));
                SVGWriter.this.write("\" y=\"");
                SVGWriter.this.write(SVGWriter.this.getY(p.getY()));
                SVGWriter.this.write("\" xlink:href=\"#point");
            }
        }
    }

    private class LineStringWriter
    extends SVGFeatureWriter {
        @Override
        protected void startElement(SimpleFeature feature) throws IOException {
            SVGWriter.this.write("<path");
        }

        @Override
        protected void startGeometry(Geometry geom) throws IOException {
            SVGWriter.this.write(" d=\"");
        }

        @Override
        protected void writeGeometry(Geometry geom) throws IOException {
            this.writePathContent(geom.getCoordinates());
        }
    }

    private class PolygonWriter
    extends SVGFeatureWriter {
        @Override
        protected void startElement(SimpleFeature feature) throws IOException {
            SVGWriter.this.write("<path");
        }

        @Override
        protected void startGeometry(Geometry geom) throws IOException {
            SVGWriter.this.write(" d=\"");
        }

        @Override
        protected void writeGeometry(Geometry geom) throws IOException {
            Polygon poly = (Polygon)geom;
            LinearRing shell = poly.getExteriorRing();
            int nHoles = poly.getNumInteriorRing();
            this.writeClosedPathContent(shell.getCoordinates());
            for (int i = 0; i < nHoles; ++i) {
                this.writeClosedPathContent(poly.getInteriorRingN(i).getCoordinates());
            }
        }
    }

    private class MultiPointWriter
    extends PointWriter {
        @Override
        protected void startElement(SimpleFeature feature) throws IOException {
            SVGWriter.this.write("<g ");
        }

        @Override
        protected void startGeometry(Geometry geom) throws IOException {
            SVGWriter.this.write("/>\n");
        }

        @Override
        protected void writeGeometry(Geometry geom) throws IOException {
            MultiPoint mp = (MultiPoint)geom;
            for (int i = 0; i < mp.getNumGeometries(); ++i) {
                super.startElement(null);
                super.writeGeometry(mp.getGeometryN(i));
                super.endGeometry(mp.getGeometryN(i));
                super.endElement(null);
            }
        }

        @Override
        protected void endElement(SimpleFeature feature) throws IOException {
            SVGWriter.this.write("</g>\n");
        }
    }

    private class MultiLineStringWriter
    extends LineStringWriter {
        @Override
        protected void writeGeometry(Geometry geom) throws IOException {
            MultiLineString ml = (MultiLineString)geom;
            for (int i = 0; i < ml.getNumGeometries(); ++i) {
                super.writeGeometry(ml.getGeometryN(i));
            }
        }
    }

    private class MultiPolygonWriter
    extends PolygonWriter {
        @Override
        protected void writeGeometry(Geometry geom) throws IOException {
            MultiPolygon mpoly = (MultiPolygon)geom;
            for (int i = 0; i < mpoly.getNumGeometries(); ++i) {
                super.writeGeometry(mpoly.getGeometryN(i));
            }
        }
    }

    private class GeometryWriter
    extends SVGFeatureWriter {
        SVGFeatureWriter delegate;

        private GeometryWriter() {
        }

        @Override
        protected void startElement(SimpleFeature feature) throws IOException {
            Geometry g = (Geometry)feature.getDefaultGeometry();
            this.delegate = null;
            if (g != null) {
                this.delegate = SVGWriter.this.writers.get(g.getClass());
            }
            if (this.delegate == null) {
                throw new IllegalArgumentException("No SVG Feature writer defined for " + String.valueOf(g));
            }
            this.delegate.startElement(feature);
        }

        @Override
        protected void startGeometry(Geometry geom) throws IOException {
            this.delegate.startGeometry(geom);
        }

        @Override
        protected void writeGeometry(Geometry geom) throws IOException {
            this.delegate.writeGeometry(geom);
        }
    }

    public class CollectSVGHandler
    extends SVGFeatureWriterHandler {
        private SVGFeatureWriter featureWriter;

        public CollectSVGHandler(SVGFeatureWriter featureWriter) {
            this.featureWriter = featureWriter;
        }

        public void writeFeature(SimpleFeature ft) throws IOException {
            this.featureWriter.writeGeometry((Geometry)ft.getDefaultGeometry());
            SVGWriter.this.write('\n');
        }
    }

    public class FIDSVGHandler
    extends SVGFeatureWriterHandler {
        private SVGFeatureWriterHandler handler;

        public FIDSVGHandler(SVGFeatureWriterHandler handler) {
            this.handler = handler;
        }

        @Override
        public void startFeature(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            this.handler.startFeature(featureWriter, ft);
            SVGWriter.this.write(" id=\"");
            try {
                SVGWriter.this.write(ft.getID());
            }
            catch (IOException ex) {
                LOGGER.severe("error getting fid from " + String.valueOf(ft));
                throw ex;
            }
            SVGWriter.this.write("\"");
        }
    }

    public class BoundsSVGHandler
    extends SVGFeatureWriterHandler {
        private SVGFeatureWriterHandler handler;

        public BoundsSVGHandler(SVGFeatureWriterHandler handler) {
            this.handler = handler;
        }

        @Override
        public void startFeature(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            this.handler.startFeature(featureWriter, ft);
            Geometry geom = (Geometry)ft.getDefaultGeometry();
            Envelope env = geom.getEnvelopeInternal();
            SVGWriter.this.write(" bounds=\"");
            SVGWriter.this.write(env.getMinX());
            SVGWriter.this.write(' ');
            SVGWriter.this.write(env.getMinY());
            SVGWriter.this.write(' ');
            SVGWriter.this.write(env.getMaxX());
            SVGWriter.this.write(' ');
            SVGWriter.this.write(env.getMaxY());
            SVGWriter.this.write('\"');
        }
    }

    public class AttributesSVGHandler
    extends SVGFeatureWriterHandler {
        private SVGFeatureWriterHandler handler;

        public AttributesSVGHandler(SVGFeatureWriterHandler handler) {
            this.handler = handler;
        }

        @Override
        public void startFeature(SVGFeatureWriter featureWriter, SimpleFeature ft) throws IOException {
            this.handler.startFeature(featureWriter, ft);
            SimpleFeatureType type = ft.getFeatureType();
            int numAtts = type.getAttributeCount();
            for (int i = 0; i < numAtts; ++i) {
                Object value = ft.getAttribute(i);
                if (value == null || value instanceof Geometry) continue;
                SVGWriter.this.write(' ');
                SVGWriter.this.write(type.getDescriptor(i).getName().getLocalPart());
                SVGWriter.this.write("=\"");
                this.encodeAttribute(String.valueOf(value));
                SVGWriter.this.write('\"');
            }
        }

        private void encodeAttribute(String inData) throws IOException {
            if (inData == null) {
                return;
            }
            int length = inData.length();
            for (int i = 0; i < length; ++i) {
                char charToCompare = inData.charAt(i);
                if (charToCompare == '\"') {
                    SVGWriter.this.write("&quot;");
                    continue;
                }
                if (charToCompare > '\u007f') {
                    this.writeUnicodeEscapeSequence(charToCompare);
                    continue;
                }
                SVGWriter.this.write(charToCompare);
            }
        }

        private void writeUnicodeEscapeSequence(char c) throws IOException {
            SVGWriter.this.write("&#x");
            String hex = Integer.toHexString(c);
            int pendingZeros = 4 - hex.length();
            for (int i = 0; i < pendingZeros; ++i) {
                SVGWriter.this.write('0');
            }
            SVGWriter.this.write(hex);
            SVGWriter.this.write(';');
        }
    }
}

