/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.shapefile;

import java.io.Closeable;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.DoubleBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.FileStore;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.geotools.data.shapefile.shp.IndexFile;
import org.geotools.data.shapefile.shp.ShapeType;
import org.geotools.data.shapefile.shp.ShapefileReader;
import org.geotools.util.NIOUtilities;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Envelope;

class ShapefileIndexerBoundsHelper {
    private static final Logger LOGGER = Logging.getLogger(ShapefileIndexerBoundsHelper.class);
    private static final int FALLBACK_TO_FILE_REC_COUNT_THRESHOLD = 32768;

    ShapefileIndexerBoundsHelper() {
    }

    static BoundsReader createBoundsReader(ShapefileReader reader, IndexFile shpIndex) {
        ShapeType shapeType;
        boolean pointBounds;
        int fileRecCountThreshold;
        int numRecs = shpIndex.getRecordCount();
        if (numRecs <= (fileRecCountThreshold = ((pointBounds = (shapeType = reader.getHeader().getShapeType()) == ShapeType.POINT || shapeType == ShapeType.POINTM || shapeType == ShapeType.POINTZ) ? 2 : 1) * 32768)) {
            return new HeapBoundsReader(numRecs, pointBounds);
        }
        try {
            return new FileBoundsReader(numRecs, pointBounds);
        }
        catch (IOException e) {
            LOGGER.log(Level.INFO, "Unable to create a temporary file backed bounds reader, falling back to slower ShapefileReader random access", e);
            return new ShapefileReaderBoundsReader(reader, shpIndex);
        }
    }

    private static class HeapBoundsReader
    extends BufferBoundsReader {
        HeapBoundsReader(int recCount, boolean pointBounds) {
            super(DoubleBuffer.allocate((pointBounds ? 2 : 4) * recCount), pointBounds);
        }
    }

    private static class FileBoundsReader
    implements BoundsReader {
        private static final String TMP_FILE_NAME_PREFIX = "GeoTools_shp_qix_bounds_";
        private final BufferBoundsReader reader;
        private final Path path;
        private final RandomAccessFile file;
        private final MappedByteBuffer mappedBuffer;

        FileBoundsReader(int recCount, boolean isPoint) throws IOException {
            int fileSize = recCount * 8 * (isPoint ? 2 : 4);
            this.path = Files.createTempFile(TMP_FILE_NAME_PREFIX, ".tmp", new FileAttribute[0]);
            try {
                this.checkAvailableSpace(this.path, fileSize);
                this.file = new RandomAccessFile(this.path.toFile(), "rw");
                this.mappedBuffer = this.file.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, fileSize);
                this.reader = new BufferBoundsReader(this.mappedBuffer.asDoubleBuffer(), isPoint);
            }
            catch (IOException | RuntimeException e) {
                Files.delete(this.path);
                throw e;
            }
        }

        @Override
        public void close() {
            this.reader.close();
            NIOUtilities.clean((ByteBuffer)this.mappedBuffer, (boolean)true);
            try {
                this.file.close();
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Error closing temporary file " + String.valueOf(this.path), e);
            }
            try {
                Files.delete(this.path);
            }
            catch (IOException e) {
                LOGGER.log(Level.WARNING, "Error deleting temporary file " + String.valueOf(this.path), e);
            }
        }

        @Override
        public void insert(int recno, Envelope env) {
            this.reader.insert(recno, env);
        }

        @Override
        public void read(int recNumber, Envelope env) throws IOException {
            this.reader.read(recNumber, env);
        }

        @Override
        public void expandEnvelope(int recNumber, Envelope env) throws IOException {
            this.reader.expandEnvelope(recNumber, env);
        }

        private void checkAvailableSpace(Path path, int fileSize) throws IOException {
            FileStore fileStore = Files.getFileStore(path);
            long usableSpace = fileStore.getUsableSpace();
            if (usableSpace < (long)fileSize) {
                throw new FileSystemException("Not enough disk space. Required %,d bytes, available: %,d ".formatted(fileSize, usableSpace));
            }
        }
    }

    private static class ShapefileReaderBoundsReader
    implements BoundsReader {
        private ShapefileReader reader;
        private IndexFile shpIndex;

        ShapefileReaderBoundsReader(ShapefileReader reader, IndexFile shpIndex) {
            this.reader = reader;
            this.shpIndex = shpIndex;
        }

        @Override
        public void insert(int recno, Envelope env) {
        }

        @Override
        public void read(int recNumber, Envelope env) throws IOException {
            int offset = this.shpIndex.getOffsetInBytes(recNumber);
            this.reader.goTo(offset);
            ShapefileReader.Record rec = this.reader.nextRecord();
            env.init(rec.minX, rec.maxX, rec.minY, rec.maxY);
        }

        @Override
        public void expandEnvelope(int recNumber, Envelope env) throws IOException {
            int offset = this.shpIndex.getOffsetInBytes(recNumber);
            this.reader.goTo(offset);
            ShapefileReader.Record rec = this.reader.nextRecord();
            env.expandToInclude(rec.minX, rec.minY);
            env.expandToInclude(rec.maxX, rec.maxY);
        }
    }

    private static class BufferBoundsReader
    implements BoundsReader {
        protected DoubleBuffer buff;
        private final boolean pointBounds;
        private final int ordinatesPerRec;

        BufferBoundsReader(DoubleBuffer buff, boolean pointBounds) {
            this.buff = buff;
            this.pointBounds = pointBounds;
            this.ordinatesPerRec = pointBounds ? 2 : 4;
        }

        private int offsetOf(int recNumber) {
            return this.ordinatesPerRec * recNumber;
        }

        @Override
        public void close() {
            this.buff = null;
        }

        @Override
        public void insert(int recno, Envelope env) {
            int offset = this.offsetOf(recno);
            this.buff.put(offset, env.getMinX());
            this.buff.put(offset + 1, env.getMinY());
            if (!this.pointBounds) {
                this.buff.put(offset + 2, env.getMaxX());
                this.buff.put(offset + 3, env.getMaxY());
            }
        }

        @Override
        public void read(int recNumber, Envelope env) throws IOException {
            this.buff.position(this.offsetOf(recNumber));
            double minx = this.buff.get();
            double miny = this.buff.get();
            if (this.pointBounds) {
                env.init(minx, minx, miny, miny);
            } else {
                double maxx = this.buff.get();
                double maxy = this.buff.get();
                env.init(minx, maxx, miny, maxy);
            }
        }

        @Override
        public void expandEnvelope(int recNumber, Envelope env) throws IOException {
            this.buff.position(this.offsetOf(recNumber));
            env.expandToInclude(this.buff.get(), this.buff.get());
            if (!this.pointBounds) {
                env.expandToInclude(this.buff.get(), this.buff.get());
            }
        }
    }

    static interface BoundsReader
    extends Closeable {
        public void insert(int var1, Envelope var2);

        public void read(int var1, Envelope var2) throws IOException;

        public void expandEnvelope(int var1, Envelope var2) throws IOException;

        @Override
        default public void close() throws IOException {
        }
    }
}

