/*
 * Decompiled with CFR 0.152.
 */
package org.geotools.data.geojson;

import com.bedatadriven.jackson.datatype.jts.JtsModule;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.PrettyPrinter;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationConfig;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.cfg.CacheProvider;
import com.fasterxml.jackson.databind.ser.DefaultSerializerProvider;
import com.fasterxml.jackson.databind.ser.SerializerFactory;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.Array;
import java.nio.charset.StandardCharsets;
import java.text.NumberFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.lang3.time.FastDateFormat;
import org.geotools.api.feature.Property;
import org.geotools.api.feature.simple.SimpleFeature;
import org.geotools.api.feature.type.GeometryType;
import org.geotools.api.feature.type.PropertyType;
import org.geotools.api.geometry.BoundingBox;
import org.geotools.api.geometry.MismatchedDimensionException;
import org.geotools.api.referencing.FactoryException;
import org.geotools.api.referencing.crs.CRSAuthorityFactory;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.api.referencing.operation.MathTransform;
import org.geotools.api.referencing.operation.TransformException;
import org.geotools.data.simple.SimpleFeatureCollection;
import org.geotools.data.simple.SimpleFeatureIterator;
import org.geotools.geometry.jts.JTS;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.referencing.CRS;
import org.geotools.referencing.operation.transform.IdentityTransform;
import org.geotools.util.logging.Logging;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;

public class GeoJSONWriter
implements AutoCloseable {
    static Logger LOGGER = Logging.getLogger((String)"org.geotools.data.geojson");
    public static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSSX";
    public static final TimeZone DEFAULT_TIME_ZONE = TimeZone.getTimeZone("GMT");
    public static final FastDateFormat DEFAULT_DATE_FORMATTER = FastDateFormat.getInstance((String)"yyyy-MM-dd'T'HH:mm:ss.SSSX", (TimeZone)DEFAULT_TIME_ZONE);
    private int maxDecimals = JtsModule.DEFAULT_MAX_DECIMALS;
    private OutputStream out;
    JsonGenerator generator;
    private ObjectMapper mapper;
    private CoordinateReferenceSystem outCRS;
    private MathTransform transform;
    private CoordinateReferenceSystem lastCRS;
    private boolean initalised = false;
    private ReferencedEnvelope bounds = null;
    private boolean encodeFeatureBounds = false;
    private boolean inArray = false;
    private boolean encodeFeatureCollectionCRS = false;
    private final JtsModule module;
    private NumberFormat formatter = NumberFormat.getNumberInstance(Locale.ENGLISH);
    private boolean notWritenBbox = true;
    private boolean singleFeature = false;
    private FastDateFormat dateFormatter = DEFAULT_DATE_FORMATTER;
    private static GeometryFactory gf = new GeometryFactory();

    public GeoJSONWriter(OutputStream outputStream) throws IOException {
        CRSAuthorityFactory cFactory = CRS.getAuthorityFactory(true);
        try {
            this.outCRS = cFactory.createCoordinateReferenceSystem("EPSG:4326");
        }
        catch (FactoryException e) {
            throw new RuntimeException("CRS factory not found in GeoJSONDatastore writer", e);
        }
        this.mapper = new ObjectMapper();
        this.module = new JtsModule(this.maxDecimals);
        this.mapper.registerModule((Module)this.module);
        this.out = outputStream instanceof BufferedOutputStream ? outputStream : new BufferedOutputStream(outputStream);
        JsonFactory factory = new JsonFactory();
        this.generator = factory.createGenerator(this.out);
    }

    private void initialise() throws IOException {
        if (!this.singleFeature) {
            this.generator.writeStartObject();
            this.generator.writeStringField("type", "FeatureCollection");
            if (this.bounds != null && this.isEncodeFeatureBounds()) {
                this.writeBoundingEnvelope();
                this.notWritenBbox = false;
            }
            this.generator.writeFieldName("features");
            this.generator.writeStartArray();
            this.inArray = true;
        }
        this.initalised = true;
    }

    private void writeBoundingEnvelope() throws IOException {
        if (!CRS.equalsIgnoreMetadata(this.bounds.getCoordinateReferenceSystem(), this.outCRS)) {
            try {
                this.bounds = this.bounds.transform(this.outCRS, true);
            }
            catch (MismatchedDimensionException | FactoryException | TransformException e) {
                throw new RuntimeException(e);
            }
        }
        this.generator.writeArrayFieldStart("bbox");
        double[] coords = new double[]{this.bounds.getMinX(), this.bounds.getMinY(), this.bounds.getMaxX(), this.bounds.getMaxY()};
        this.formatter.setMaximumFractionDigits(this.maxDecimals);
        for (double c : coords) {
            this.generator.writeNumber(this.formatter.format(c));
        }
        this.generator.writeEndArray();
    }

    public void write(SimpleFeature currentFeature) throws IOException {
        if (!this.initalised) {
            this.initialise();
        }
        this.writeFeature(currentFeature, this.generator);
    }

    private void writeFeature(SimpleFeature currentFeature, JsonGenerator g) throws IOException, JsonProcessingException {
        Geometry defaultGeometry = (Geometry)currentFeature.getDefaultGeometry();
        if (this.isEncodeFeatureBounds() && this.notWritenBbox) {
            BoundingBox bbox = currentFeature.getDefaultGeometryProperty().getBounds();
            if (this.bounds == null) {
                this.bounds = new ReferencedEnvelope(bbox);
            } else {
                this.bounds.expandToInclude((Envelope)bbox);
            }
        }
        g.writeStartObject();
        g.writeStringField("type", "Feature");
        g.writeFieldName("properties");
        g.writeStartObject();
        for (Property p : currentFeature.getProperties()) {
            PropertyType type = p.getType();
            if (type instanceof GeometryType) continue;
            Object value = p.getValue();
            String name = p.getName().getLocalPart();
            if (value == null) {
                g.writeNullField(name);
                continue;
            }
            Class binding = p.getType().getBinding();
            g.writeFieldName(name);
            this.writeValue(g, value, binding);
        }
        g.writeEndObject();
        if (defaultGeometry != null) {
            if (!this.encodeFeatureCollectionCRS) {
                defaultGeometry = this.reprojectGeometry(currentFeature);
            }
            if (this.isEncodeFeatureBounds()) {
                this.writeBbox(g, defaultGeometry);
            }
            g.writeFieldName("geometry");
            String gString = this.mapper.writeValueAsString((Object)defaultGeometry);
            g.writeRawValue(gString);
        } else {
            g.writeFieldName("geometry");
            g.writeNull();
        }
        g.writeStringField("id", currentFeature.getID());
        g.writeEndObject();
        g.flush();
    }

    private void writeValue(JsonGenerator g, Object value, Class<?> binding) throws IOException {
        if (value == null) {
            g.writeNull();
            return;
        }
        if (binding == Integer.class) {
            g.writeNumber(((Integer)value).intValue());
        } else if (binding == Double.class) {
            g.writeNumber(((Double)value).doubleValue());
        } else if (binding == Boolean.class) {
            g.writeBoolean(((Boolean)value).booleanValue());
        } else if (Date.class.isAssignableFrom(binding)) {
            g.writeString(this.dateFormatter.format(value));
        } else if (Object.class.isAssignableFrom(binding) && value instanceof JsonNode) {
            JsonNode node = (JsonNode)value;
            node.serialize(g, (SerializerProvider)new DefaultSerializerProvider(this.mapper.getSerializerProvider(), this.mapper.getSerializationConfig(), this.mapper.getSerializerFactory()){

                public DefaultSerializerProvider createInstance(SerializationConfig config, SerializerFactory jsf) {
                    throw new UnsupportedOperationException();
                }

                public DefaultSerializerProvider withCaches(CacheProvider cacheProvider) {
                    throw new UnsupportedOperationException();
                }
            });
        } else if (binding.isArray()) {
            g.writeStartArray();
            int length = Array.getLength(value);
            for (int i = 0; i < length; ++i) {
                this.writeValue(g, Array.get(value, i), binding.getComponentType());
            }
            g.writeEndArray();
        } else if (Collection.class.isAssignableFrom(binding)) {
            g.writeStartArray();
            for (Object v : (Collection)value) {
                this.writeValue(g, v, v == null ? null : v.getClass());
            }
            g.writeEndArray();
        } else {
            g.writeString(value.toString());
        }
    }

    private Geometry reprojectGeometry(SimpleFeature currentFeature) {
        Geometry defaultGeometry = (Geometry)currentFeature.getDefaultGeometry();
        if (defaultGeometry == null) {
            LOGGER.fine("No geometry found in " + currentFeature.getID() + " skipping");
            GeometryCollection collection = gf.createGeometryCollection(null);
            return collection;
        }
        CoordinateReferenceSystem inCRS = currentFeature.getDefaultGeometryProperty().getDescriptor().getCoordinateReferenceSystem();
        if (this.transform == null || inCRS != this.lastCRS) {
            this.lastCRS = inCRS;
            try {
                this.transform = inCRS == null ? IdentityTransform.create(2) : CRS.findMathTransform(inCRS, this.outCRS, true);
            }
            catch (FactoryException e) {
                throw new RuntimeException(e);
            }
        }
        if (!CRS.equalsIgnoreMetadata(inCRS, this.outCRS)) {
            try {
                defaultGeometry = JTS.transform(defaultGeometry, this.transform);
            }
            catch (MismatchedDimensionException | TransformException e) {
                throw new RuntimeException(e);
            }
        }
        return defaultGeometry;
    }

    private void writeBbox(JsonGenerator g, Geometry defaultGeometry) throws IOException {
        g.writeFieldName("bbox");
        Envelope envelope = defaultGeometry.getEnvelopeInternal();
        double[] coords = new double[]{envelope.getMinX(), envelope.getMinY(), envelope.getMaxX(), envelope.getMaxY()};
        this.formatter.setMaximumFractionDigits(this.maxDecimals);
        g.writeStartArray();
        for (double c : coords) {
            g.writeNumber(this.formatter.format(c));
        }
        g.writeEndArray();
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.inArray) {
                this.generator.writeEndArray();
                if (this.isEncodeFeatureBounds() && this.notWritenBbox) {
                    this.writeBoundingEnvelope();
                }
                this.generator.writeEndObject();
            }
            this.generator.close();
        }
        finally {
            this.out.close();
        }
    }

    public boolean isEncodeFeatureBounds() {
        return this.encodeFeatureBounds;
    }

    public void setEncodeFeatureBounds(boolean encodeFeatureCollectionBounds) {
        this.encodeFeatureBounds = encodeFeatureCollectionBounds;
    }

    public static String toGeoJSON(Geometry geometry) {
        return GeoJSONWriter.toGeoJSON(geometry, JtsModule.DEFAULT_MAX_DECIMALS);
    }

    public static String toGeoJSON(Geometry geometry, int maxDecimals) {
        ObjectMapper lMapper = new ObjectMapper();
        lMapper.registerModule((Module)new JtsModule(maxDecimals));
        try {
            return lMapper.writeValueAsString((Object)geometry);
        }
        catch (JsonProcessingException e) {
            LOGGER.warning(e.getLocalizedMessage());
            return "";
        }
    }

    public static String toGeoJSON(SimpleFeature f) {
        JsonFactory factory = new JsonFactory();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try (JsonGenerator lGenerator = factory.createGenerator((OutputStream)out);
             GeoJSONWriter writer = new GeoJSONWriter(out);){
            writer.writeFeature(f, lGenerator);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
        return new String(out.toByteArray(), StandardCharsets.UTF_8);
    }

    public static String toGeoJSON(SimpleFeatureCollection fc) {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        try (GeoJSONWriter writer = new GeoJSONWriter(out);
             SimpleFeatureIterator itr = fc.features();){
            while (itr.hasNext()) {
                SimpleFeature f = (SimpleFeature)itr.next();
                writer.write(f);
            }
        }
        catch (IOException e) {
            LOGGER.warning("Unexpected IOException converting featureCollection to GeoJSON");
            LOGGER.log(Level.FINE, "Unexpected IOException converting featureCollection to GeoJSON", e);
        }
        return new String(out.toByteArray(), StandardCharsets.UTF_8);
    }

    public void setEncodeFeatureCollectionCRS(boolean b) {
        this.encodeFeatureCollectionCRS = b;
    }

    public boolean isEncodeFeatureCollectionCRS() {
        return this.encodeFeatureCollectionCRS;
    }

    public int getMaxDecimals() {
        return this.maxDecimals;
    }

    public void setMaxDecimals(int number) {
        this.maxDecimals = number;
        this.module.setMaxDecimals(number);
    }

    public void writeFeatureCollection(SimpleFeatureCollection features) throws IOException {
        if (!this.initalised) {
            this.initialise();
        }
        try (SimpleFeatureIterator itr = features.features();){
            while (itr.hasNext()) {
                SimpleFeature feature = (SimpleFeature)itr.next();
                this.write(feature);
            }
        }
    }

    public void setBounds(ReferencedEnvelope bbox) {
        this.bounds = bbox;
    }

    public boolean isSingleFeature() {
        return this.singleFeature;
    }

    public void setSingleFeature(boolean singleFeature) {
        this.singleFeature = singleFeature;
    }

    public void setPrettyPrinting(boolean prettyPrint) {
        if (prettyPrint) {
            this.generator.setPrettyPrinter((PrettyPrinter)new DefaultPrettyPrinter());
        } else {
            this.generator.setPrettyPrinter(null);
        }
    }

    public boolean isPrettyPrinting() {
        return this.generator.getPrettyPrinter() != null;
    }

    public void setTimeZone(TimeZone tz) {
        this.dateFormatter = FastDateFormat.getInstance((String)this.dateFormatter.getPattern(), (TimeZone)tz);
    }

    public TimeZone getTimeZone(TimeZone tz) {
        return this.dateFormatter.getTimeZone();
    }

    public void setDatePattern(String pattern) {
        this.dateFormatter = FastDateFormat.getInstance((String)pattern, (TimeZone)this.dateFormatter.getTimeZone());
    }

    public String getDatePattern() {
        return this.dateFormatter.getPattern();
    }
}

